package easik.sketch.attribute;

import java.util.Iterator;

import javax.swing.tree.DefaultMutableTreeNode;

import easik.Easik;
import easik.sketch.datatype.DataType;
import easik.sketch.util.Export.ExportConstants;
import easik.sketch.vertex.EntityNode;


/**
 * This class is used to store information related to entity attributes.
 * 
 * @author Kevin Green 2006
 * @version 2006-07-13 Kevin Green
 */
public class EntityAttribute {
	/**
	 * The name of the attribute
	 */
	private String _name;
	/**
	 * The data type of the attribute
	 */
	private DataType _dataType;
	/**
	 * The temporary data type
	 */
	private String _tempDataType;
	
	/**
	 * The representation of this attribute as a DefaultMutableTreeNode
	 */
	private DefaultMutableTreeNode _attNode;
	
	/**
	 * Default Constructor
	 * 
	 * @param inName Name of attribute
	 * @param inDataType Data type of attribute
	 */
	public EntityAttribute(String inName, DataType inDataType){
		_name = inName;
		_dataType = inDataType;
	}
	
	/**
	 * Constructor for reading in from file when the data types have
	 * not yet been initiated and thus we must load their references 
	 * after loading them into the data type controller
	 * 
	 * @param inName The name of the attribute
	 * @param inType The name of the data type used for the attribute
	 */
	public EntityAttribute(String inName, String inType){
		_name = inName;
		_tempDataType = inType;
	}
	
	/**
	 * Overwrite standard toString function.
	 * (provides functionality for inserting attributes to the tree)
	 * @return The name of this attribute formatted for display on the tree
	 */
	public String toString(){
		return _name + " {" + getDataType().getTypeName() + "}";
	}
	
	/**
	 * Sets the attribute name.
	 * 
	 * @param inName Attribute Name
	 */
	public void setName(String inName){
		_name = inName;
	}
	
	/**
	 * Returns name of attribute.
	 * 
	 * @return Name of attribute.
	 */
	public String getName(){
		return _name;
	}
	
	/**
	 * Returns data type
	 * 
	 * @return Data type
	 */
	public DataType getDataType(){
		if(_dataType == null){
			setTypeFromTemp();
		}
		return _dataType;
	}
	
	/**
	 * Sets the data type from the temporary type name
	 */
	public void setTypeFromTemp(){
		Iterator it = Easik.getInstance().getFrame().getSketch().getDataTypeController().getDataTypes().iterator();
		while(it.hasNext()){
			DataType curType = (DataType) it.next();
			if(curType.getTypeName().equals(_tempDataType)){
				_tempDataType = null;
				_dataType = curType;
				return;
			}
		}
	}
	
	/**
	 * Sets the data type of the attribute
	 * 
	 * @param inType The data type of the attribute
	 */
	public void setDataType(DataType inType){
		_dataType = inType;
	}
	
	/**
	 * Sets the node used for displaying the attribute
	 * 
	 * @param inNode The node used for displaying the attribute
	 */
	public void setNode(DefaultMutableTreeNode inNode){
		_attNode = inNode;
	}
	
	/**
	 * Returns the node used for displaying the attribute
	 * 
	 * @return The node used for displaying the attribute
	 */
	public DefaultMutableTreeNode getNode(){
		return _attNode;
	}
	
	/**
	 * Check to see if an attribute name is already used in the entity
	 * 
	 * @param inNode The entity being checked against
	 * @param attName The name being checked for duplication
	 * @return true if the name is in use, false otherwise
	 */
	public static boolean isAttNameUsed(EntityNode inNode, String attName){
		//Loop through all attributes of the entity
		for(int i=0; i<inNode.getAttributes().size(); i++){
			EntityAttribute curAtt = (EntityAttribute)inNode.getAttributes().get(i);
			if(curAtt.getName().toUpperCase().equals(attName.toUpperCase()))
				return true;
		}
		
		//No duplicate found
		return false;
	}
	/**
	 * Determines whether an attribute name is protected.  Attribute names that commence with the name of the entity
	 * followed by the SQL identifier are protected
	 * @param parent The entity node to which the attribute is being added.
	 * @param attName The potential name of the attribute
	 * @return true if name is protected, false otherwise
	 * @since 2006-08-03 Vera Ranieri
	 */
	public static boolean isProtectedName(EntityNode parent, String attName){
		String parentName = parent.getName().trim() + ExportConstants.ID.trim();
		attName = attName.trim();
		if(attName.equalsIgnoreCase(parentName) || attName.equalsIgnoreCase(parentName.replace('_', ' '))){
			
			return true;
		}
		return false;
	}
}
